<?php

/**
 * Importer - Instant winner Rule.
 *
 * @since 9.9.0
 */
defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'LTY_Instant_Winner_Rule_Importer' ) ) {

	/**
	 * Class.
	 *
	 * @since 9.9.0
	 */
	class LTY_Instant_Winner_Rule_Importer extends LTY_Importer {

		/**
		 * Action type.
		 *
		 * @since 9.9.0
		 * @var string
		 */
		protected $action_type = 'instant-winner-rule';

		/**
		 * Product.
		 *
		 * @since 9.9.0
		 * @var object
		 */
		private $product;

		/**
		 * Get the popup header label.
		 *
		 * @since 9.9.0
		 * @return string
		 */
		public function get_popup_header_label() {
			return __( 'Import for Instant Win Prizes Rules', 'lottery-for-woocommerce' );
		}

		/**
		 * Get upload file description.
		 *
		 * @since 9.9.0
		 * @return string
		 */
		protected function get_upload_file_description() {
			return __( 'If you import new instant win data, then no need to mention the instant win field id and product id in the CSV file. If you want to update the existing data, you should mention the instant win field id in the CSV file.', 'lottery-for-woocommerce' );
		}

		/**
		 * Get the map columns.
		 *
		 * @since 9.9.0
		 * @return array
		 */
		public function get_map_columns() {
			return array(
				__( 'ID', 'lottery-for-woocommerce' )    => 'id',
				__( 'Image ID', 'lottery-for-woocommerce' ) => 'image_id',
				__( 'Ticket Number', 'lottery-for-woocommerce' ) => 'ticket_number',
				__( 'Prize Type', 'lottery-for-woocommerce' ) => 'prize_type',
				__( 'Coupon Generation Type', 'lottery-for-woocommerce' ) => 'coupon_generation_type',
				__( 'Coupon Discount Type', 'lottery-for-woocommerce' ) => 'coupon_discount_type',
				__( 'Coupon ID', 'lottery-for-woocommerce' ) => 'coupon_id',
				__( 'Prize Amount', 'lottery-for-woocommerce' ) => 'prize_amount',
				__( 'Prize', 'lottery-for-woocommerce' ) => 'prize',
			);
		}

		/**
		 * Process item.
		 *
		 * @since 9.9.0
		 */
		protected function process_item( $item ) {
			$wp_error = $this->validate_item( $item );
			if ( ! empty( $wp_error ) ) {
				return $wp_error;
			}

			$result = array();

			try {
				$id                  = isset( $item['id'] ) ? absint( $item['id'] ) : '';
				$instant_winner_rule = lty_get_instant_winner_rule( $id );
				$relist_count        = is_callable( array( $this->get_product(), 'get_current_relist_count' ) ) ? $this->get_product()->get_current_relist_count() : 0;
				$args                = array(
					'lty_image_id'               => $item['image_id'],
					'lty_ticket_number'          => $item['ticket_number'],
					'lty_prize_type'             => $item['prize_type'],
					'lty_coupon_generation_type' => $item['coupon_generation_type'],
					'lty_coupon_discount_type'   => $item['coupon_discount_type'],
					'lty_coupon_id'              => $item['coupon_id'],
					'lty_prize_amount'           => $item['prize_amount'],
					'lty_instant_winner_prize'   => $item['prize'],
				);

				if ( $instant_winner_rule->exists() ) {
					$instant_winner_log_id = lty_get_instant_winner_log_id_by_rule_id( $instant_winner_rule->get_id(), $relist_count );
					if ( ! $instant_winner_log_id ) {
						throw new Exception( esc_html__( 'Instant Winner log does not exists', 'lottery-for-woocommerce' ) );
					}

					lty_update_instant_winner_rule( $id, $args );
					lty_update_instant_winner_log( $instant_winner_log_id, array_merge( $args, array( 'lty_current_relist_count' => $relist_count ) ), array( 'post_parent' => $instant_winner_rule->get_id() ) );

					$result['updated'] = $instant_winner_rule->get_id();
				} else {
					$instant_winner_rule_id = lty_create_new_instant_winner_rule( $args, array( 'post_parent' => $this->get_product()->get_id() ) );
					lty_create_new_instant_winner_log( array_merge( $args, array( 'lty_current_relist_count' => $relist_count ) ), array( 'post_parent' => $instant_winner_rule_id ) );

					$result['imported'] = $instant_winner_rule_id;
				}
			} catch ( Exception $ex ) {
				$result = new WP_Error( 'lty_instant_winner_rule_importer_error', $ex->getMessage() );
			}

			return $result;
		}

		/**
		 * Validate item.
		 *
		 * @since 9.9.0
		 */
		protected function validate_item( $item ) {
			if ( ! $this->get_product() ) {
				return new WP_Error( 'lty_instant_winner_rule_importer_error', esc_html__( 'Product ID does not valid', 'woocommerce' ) );
			}

			$id                  = isset( $item['id'] ) ? absint( $item['id'] ) : '';
			$instant_winner_rule = lty_get_instant_winner_rule( $id );
			// Validate if the ID does not exists when enabled update existing data.
			if ( ! empty( $id ) && ! $instant_winner_rule->exists() ) {
				return new WP_Error( 'lty_instant_winner_rule_importer_error', esc_html__( 'Instant Win Id is not match', 'woocommerce' ) );
			}

			// Validate if the ticket number is required.
			if ( empty( $item['ticket_number'] ) ) {
				return new WP_Error( 'lty_instant_winner_rule_importer_error', esc_html__( 'Ticket Number is empty', 'woocommerce' ) );
			}

			$rule_id = lty_get_rule_id_by_ticket_number( $this->get_product()->get_id(), $item['ticket_number'] );
			// Validate if the ticket number is already used.
			if ( $rule_id && $id !== $rule_id ) {
				return new WP_Error( 'lty_instant_winner_rule_importer_error', esc_html__( 'Ticket Number is already exists', 'woocommerce' ) );
			}

			// Validate if the prize type is required.
			if ( empty( $item['prize_type'] ) ) {
				return new WP_Error( 'lty_instant_winner_rule_importer_error', esc_html__( 'Prize Type value is empty', 'woocommerce' ) );
			}

			// Validate if the existing coupon id is empty.
			if ( 'coupon' === $item['prize_type'] && '2' === $item['coupon_generation_type'] && empty( $item['coupon_id'] ) ) {
				return new WP_Error( 'lty_instant_winner_rule_importer_error', esc_html__( 'Coupon ID is empty', 'woocommerce' ) );
			}

			// Validate if the prize amount is empty.
			if ( 'physical' !== $item['prize_type'] && empty( $item['prize_amount'] ) ) {
				return new WP_Error( 'lty_instant_winner_rule_importer_error', esc_html__( 'Prize Amount is empty', 'woocommerce' ) );
			}

			// Validate if the prize is required.
			if ( empty( $item['prize'] ) ) {
				return new WP_Error( 'lty_instant_winner_rule_importer_error', esc_html__( 'Prize value is empty', 'woocommerce' ) );
			}

			return false;
		}

		/**
		 * Get the product.
		 *
		 * @since 9.9.0
		 * @return string
		 */
		private function get_product() {
			if ( isset( $this->product ) ) {
				return $this->product;
			}

			$this->product = wc_get_product( $this->get_product_id() );

			return $this->product;
		}

		/**
		 * Get the product ID.
		 *
		 * @since 9.9.0
		 * @return string
		 */
		private function get_product_id() {
			$extra_data = $this->get_extra_data();

			return isset( $extra_data['product_id'] ) ? $extra_data['product_id'] : '';
		}
	}

}
