<?php

/**
 * Post functions.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! function_exists( 'lty_create_new_lottery_ticket' ) ) {

	/**
	 * Create New Lottery Ticket.
	 *
	 * @return integer/string
	 * */
	function lty_create_new_lottery_ticket( $meta_args, $post_args = array() ) {

		$object = new LTY_Lottery_Ticket();
		$id     = $object->create( $meta_args, $post_args );

		return $id;
	}
}

if ( ! function_exists( 'lty_get_lottery_ticket' ) ) {

	/**
	 * Get lottery ticket object.
	 *
	 * @return object
	 * */
	function lty_get_lottery_ticket( $id ) {
		$object = new LTY_Lottery_Ticket( $id );

		return $object;
	}
}

if ( ! function_exists( 'lty_update_lottery_ticket' ) ) {

	/**
	 * Update lottery ticket.
	 *
	 * @return object
	 * */
	function lty_update_lottery_ticket( $id, $meta_args, $post_args = array() ) {

		$object = new LTY_Lottery_Ticket( $id );
		$object->update( $meta_args, $post_args );

		return $object;
	}
}

if ( ! function_exists( 'lty_delete_lottery_ticket' ) ) {

	/**
	 * Delete lottery ticket.
	 *
	 * @return bool
	 * */
	function lty_delete_lottery_ticket( $id, $force = true ) {

		wp_delete_post( $id, $force );

		return true;
	}
}

if ( ! function_exists( 'lty_get_lottery_ticket_statuses' ) ) {

	/**
	 * Get Lottery Ticket statuses.
	 *
	 * @return array
	 * */
	function lty_get_lottery_ticket_statuses() {
		/**
		 * This hook is used to alter the lottery ticket statuses.
		 *
		 * @since 1.0
		 */
		return apply_filters( 'lty_lottery_ticket_statuses', array( 'lty_ticket_buyer', 'lty_ticket_winner' ) );
	}
}

if ( ! function_exists( 'lty_get_ticket_statuses' ) ) {

	/**
	 * Get the ticket statuses.
	 *
	 * @return array
	 * */
	function lty_get_ticket_statuses() {
		/**
		 * This hook is used to alter the ticket statuses.
		 *
		 * @since 1.0
		 */
		return apply_filters( 'lty_ticket_statuses', array( 'lty_ticket_pending', 'lty_ticket_buyer', 'lty_ticket_winner' ) );
	}
}

if ( ! function_exists( 'lty_create_new_lottery_winner' ) ) {

	/**
	 * Create New Lottery Winner.
	 *
	 * @return integer/string
	 * */
	function lty_create_new_lottery_winner( $meta_args, $post_args = array() ) {
		$object = new LTY_Lottery_Product_Winner();
		$id     = $object->create( $meta_args, $post_args );

		return $id;
	}
}

if ( ! function_exists( 'lty_get_lottery_winner' ) ) {

	/**
	 * Get lottery winner object.
	 *
	 * @return object
	 * */
	function lty_get_lottery_winner( $id ) {
		$object = new LTY_Lottery_Product_Winner( $id );

		return $object;
	}
}

if ( ! function_exists( 'lty_update_lottery_winner' ) ) {

	/**
	 * Update lottery winner.
	 *
	 * @return object
	 * */
	function lty_update_lottery_winner( $id, $meta_args, $post_args = array() ) {
		$object = new LTY_Lottery_Product_Winner( $id );
		$object->update( $meta_args, $post_args );

		return $object;
	}
}

if ( ! function_exists( 'lty_delete_lottery_winner' ) ) {

	/**
	 * Delete lottery winner.
	 *
	 * @return bool
	 * */
	function lty_delete_lottery_winner( $id, $force = true ) {

		wp_delete_post( $id, $force );

		return true;
	}
}

if ( ! function_exists( 'lty_get_lottery_tickets' ) ) {

	/**
	 * Get the lottery tickets.
	 *
	 * @return array
	 * */
	function lty_get_lottery_tickets( $user_id = false, $product_id = false, $from_date = false, $end_date = false, $post_status = false, $order_by = 'ID', $order = 'DESC', $page = false, $limit = false ) {

		$args = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE,
			'post_status'    => lty_get_lottery_ticket_statuses(),
			'posts_per_page' => '-1',
			'fields'         => 'ids',
			'orderby'        => 'ID',
		);

		if ( false !== $user_id ) {
			$args['meta_key']   = 'lty_user_id';
			$args['meta_value'] = $user_id;
		}

		if ( false !== $product_id ) {
			$args['post_parent'] = lty_get_lottery_id( $product_id );
		} else {
			$args['post_parent__in'] = lty_get_lottery_ids();
		}

		if ( false !== $from_date || false !== $end_date ) {
			$args['date_query'] = array();

			if ( false !== $from_date ) {
				$args['date_query'][] = array(
					'column' => 'post_date_gmt',
					'after'  => $from_date,
				);
			}

			if ( false !== $end_date ) {
				$args['date_query'][] = array(
					'column' => 'post_date_gmt',
					'before' => $end_date,
				);
			}
		}

		if ( lty_check_is_array( $post_status ) ) {
			$args['post_status'] = $post_status;
		} elseif ( false !== $post_status ) {
			$args['post_status'] = array_merge( array( 'lty_ticket_pending' ), $args['post_status'] );
		}

		if ( 'ID' != $order_by ) {
			$args['meta_key'] = 'lty_ticket_number';
			$args['orderby']  = $order_by;
		}

		if ( $page ) {
			$args['paged'] = $page;
		}

		if ( $limit ) {
			$args['posts_per_page'] = $limit;
		}

		$args['order'] = $order;

		return get_posts( $args );
	}
}

if ( ! function_exists( 'lty_get_valid_winner_lottery_tickets' ) ) {

	/**
	 * Get the valid winner lottery tickets.
	 *
	 * @return array
	 * */
	function lty_get_valid_winner_lottery_tickets( $product ) {
		if ( ! lty_is_lottery_product( $product ) ) {
			return array();
		}

		$args = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE,
			'post_status'    => 'lty_ticket_buyer',
			'posts_per_page' => '-1',
			'fields'         => 'ids',
			'orderby'        => 'ID',
			'post_parent'    => $product->get_lottery_id(),
		);

		if ( false !== $product->get_lty_relisted_date_gmt() ) {
			$args['date_query'][] = array(
				'column' => 'post_date_gmt',
				'after'  => $product->get_lty_relisted_date_gmt(),
			);
		}

		return get_posts( $args );
	}
}

if ( ! function_exists( 'lty_get_lottery_winners' ) ) {

	/**
	 * Get lottery winners.
	 *
	 * @return array
	 * */
	function lty_get_lottery_winners( $user_id = false, $product_id = false, $from_date = false, $end_date = false ) {

		$args = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_WINNER_POSTTYPE,
			'post_status'    => array( 'lty_publish' ),
			'posts_per_page' => '-1',
			'fields'         => 'ids',
			'orderby'        => 'ID',
		);

		if ( false !== $user_id ) {
			$args['meta_key']   = 'lty_user_id';
			$args['meta_value'] = $user_id;
		}

		if ( false !== $product_id ) {
			$args['post_parent'] = lty_get_lottery_id( $product_id );
		}

		if ( false !== $from_date || false !== $end_date ) {
			$args['date_query'] = array();

			if ( false !== $from_date ) {
				$args['date_query'][] = array(
					'column'    => 'post_date_gmt',
					'after'     => $from_date,
					'inclusive' => true,
				);
			}

			if ( false !== $end_date ) {
				$args['date_query'][] = array(
					'column'    => 'post_date_gmt',
					'before'    => $end_date,
					'inclusive' => true,
				);
			}
		}

		return get_posts( $args );
	}
}

if ( ! function_exists( 'lty_get_lottery_statuses' ) ) {

	/**
	 * Get Lottery statuses.
	 *
	 * @return array
	 * */
	function lty_get_lottery_statuses() {
		static $lottery_statuses;

		if ( $lottery_statuses ) {
			return $lottery_statuses;
		}
		/**
		 * This hook is used to alter the lottery statuses.
		 *
		 * @since 1.0
		 */
		return apply_filters(
			'lty_lottery_statuses',
			array(
				'lty_lottery_not_started'     => get_option( 'lty_settings_lottery_not_started_label', 'Lottery Not Started' ),
				'lty_lottery_started'         => get_option( 'lty_settings_lottery_started_label', 'Lottery Started' ),
				'lty_lottery_closed'          => get_option( 'lty_settings_lottery_closed_label', 'Lottery Closed' ),
				'lty_lottery_pending_payment' => get_option( 'lty_settings_lottery_pending_payment_label', 'Lottery Pending Payment' ),
				'lty_lottery_finished'        => get_option( 'lty_settings_lottery_finished_label', 'Lottery Finished' ),
				'lty_lottery_failed'          => get_option( 'lty_settings_lottery_failed_label', 'Lottery Failed' ),
			)
		);
	}
}

if ( ! function_exists( 'lty_lottery_reset_meta_keys' ) ) {

	/**
	 * Get lottery reset meta keys
	 *
	 * @return array
	 */
	function lty_lottery_reset_meta_keys() {
		/**
		 * This hook is used to alter the lottery reset meta keys.
		 *
		 * @since 1.0
		 */
		return apply_filters(
			'lty_lottery_reset_meta_keys',
			array(
				'_lty_closed',
				'_lty_failed_reason',
				'_lty_closed_date',
				'_lty_closed_date_gmt',
				'_lty_failed_date',
				'_lty_failed_date_gmt',
				'_lty_last_activity',
				'_lty_ticket_count',
				'_lty_manual_reserved_tickets',
				'_lty_incorrect_answer_user_ids',
				'_lty_question_answer_viewed_data',
				'_lty_question_answer_attempts_data',
				'_lty_hold_tickets',
			)
		);
	}
}

if ( ! function_exists( 'lty_lottery_extend_meta_keys' ) ) {

	/**
	 * Get the lottery extend meta keys.
	 *
	 * @return array
	 */
	function lty_lottery_extend_meta_keys() {
		/**
		 * This hook is used to alter the lottery extend meta keys.
		 *
		 * @since 1.0
		 */
		return apply_filters(
			'lty_lottery_extend_meta_keys',
			array(
				'_lty_closed',
				'_lty_failed_reason',
				'_lty_closed_date',
				'_lty_closed_date_gmt',
				'_lty_failed_date',
				'_lty_failed_date_gmt',
			)
		);
	}
}

if ( ! function_exists( 'lty_get_failed_reasons' ) ) {

	/**
	 * Get lottery failed reasons.
	 *
	 * @return array
	 */
	function lty_get_failed_reasons() {
		static $reasons;

		if ( ! $reasons ) {
			/**
			 * This hook is used to alter the lottery failed reasons.
			 *
			 * @since 1.0
			 */
			$reasons = apply_filters(
				'lty_lottery_failed_reason',
				array(
					'1' => get_option( 'lty_settings_minimum_tickets_not_met_error', __( 'Minimum Lottery Ticket not met', 'lottery-for-woocommerce' ) ),
					'2' => get_option( 'lty_settings_unique_winners_count_not_met_error', __( 'Unique Winners Count not met', 'lottery-for-woocommerce' ) ),
				)
			);
		}

		return $reasons;
	}
}

if ( ! function_exists( 'lty_get_lottery_ticket_ids_by_product_id' ) ) {

	/**
	 * Get User Lottery Ticket id's
	 *
	 * @return array.
	 */
	function lty_get_lottery_ticket_ids_by_product_id( $product_id, $count = false, $status = false ) {
		if ( empty( $product_id ) ) {
			return;
		}

		$lottery_tickets = lty_get_lottery_tickets( false, $product_id, false, false, $status );

		if ( $count ) {
			return count( $lottery_tickets );
		}

		return $lottery_tickets;
	}
}

if ( ! function_exists( 'lty_get_lottery_ticket_ids_by_user_id' ) ) {

	/**
	 * Get Lottery Ticket id's from user ID
	 *
	 * @return array.
	 */
	function lty_get_lottery_ticket_ids_by_user_id( $user_id, $count = false ) {
		if ( empty( $product_id ) ) {
			return;
		}

		if ( empty( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		$lottery_tickets = lty_get_lottery_tickets( $user_id );

		if ( $count ) {
			return count( $lottery_tickets );
		}

		return $lottery_tickets;
	}
}


if ( ! function_exists( 'lty_product_ticket_number_exists' ) ) {

	/**
	 * Check the product ticket number exists.
	 *
	 * @return bool.
	 */
	function lty_product_ticket_number_exists( $product_id, $ticket_number, $post_status = true ) {

		return lty_product_ticket_numbers_exists( $product_id, array( $ticket_number ), $post_status );
	}
}

if ( ! function_exists( 'lty_product_ticket_numbers_exists' ) ) {

	/**
	 * Check the product ticket numbers exists.
	 *
	 * @return bool.
	 */
	function lty_product_ticket_numbers_exists( $product_id, $ticket_numbers, $post_status = true ) {
		if ( empty( $product_id ) || empty( $ticket_numbers ) ) {
			return true;
		}

		$product = wc_get_product( $product_id );

		if ( ! is_object( $product ) ) {
			return true;
		}

		$post_statuses = lty_get_lottery_ticket_statuses();

		if ( $post_status ) {
			$post_statuses = array_merge( array( 'lty_ticket_pending' ), $post_statuses );
		}

		global $wpdb;
		$post_query = new LTY_Query( $wpdb->prefix . 'postmeta', 'pm' );
		$post_query->select( 'DISTINCT `p`.ID' )
				->leftJoin( $wpdb->prefix . 'posts', 'p', '`p`.`ID` = `pm`.`post_id`' )
				->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE )
				->whereIn( '`p`.post_status', $post_statuses )
				->where( '`p`.post_parent', $product->get_lottery_id() )
				->where( '`pm`.meta_key', 'lty_ticket_number' )
				->whereIn( '`pm`.meta_value', $ticket_numbers )
				->whereBetween( '`p`.post_date_gmt', $product->get_current_start_date_gmt(), $product->get_lty_end_date_gmt() );

		$lottery_ticket_ids = $post_query->fetchArray();

		if ( lty_check_is_array( $lottery_ticket_ids ) ) {
			return true;
		}

		return false;
	}
}

if ( ! function_exists( 'lty_get_user_placed_ticket_numbers_by_product_id' ) ) {

	/**
	 * Get the lottery ticket numbers by user placed using product id.
	 *
	 * @return array.
	 */
	function lty_get_user_placed_ticket_numbers_by_product_id( $product = false, $user_id = false ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
			if ( ! $user_id ) {
				return array();
			}
		}

		return lty_get_ticket_numbers( $product, $user_id );
	}
}

if ( ! function_exists( 'lty_get_user_purchased_ticket_numbers_by_product_id' ) ) {

	/**
	 * Get the lottery ticket numbers by user purchased using product id.
	 *
	 * @return array.
	 */
	function lty_get_user_purchased_ticket_numbers_by_product_id( $product = false, $user_id = false ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
			if ( ! $user_id ) {
				return array();
			}
		}

		return lty_get_ticket_numbers( $product, $user_id, false );
	}
}

if ( ! function_exists( 'lty_get_placed_ticket_numbers_by_product_id' ) ) {

	/**
	 * Get the lottery ticket numbers placed using product id.
	 *
	 * @return array.
	 */
	function lty_get_placed_ticket_numbers_by_product_id( $product = false ) {

		return lty_get_ticket_numbers( $product );
	}
}

if ( ! function_exists( 'lty_get_purchased_ticket_numbers_by_product_id' ) ) {

	/**
	 * Get the lottery ticket numbers purchased using product id.
	 *
	 * @return array.
	 */
	function lty_get_purchased_ticket_numbers_by_product_id( $product = false ) {
		return lty_get_ticket_numbers( $product, false, false );
	}
}

if ( ! function_exists( 'lty_get_purchased_ticket_numbers_by_order' ) ) {

	/**
	 * Get the lottery ticket numbers purchased by order.
	 *
	 * @return array.
	 */
	function lty_get_purchased_ticket_numbers_by_order( $product, $order_id, $user_id ) {
		return lty_get_ticket_numbers( $product, $user_id, 'all', $order_id );
	}
}

if ( ! function_exists( 'lty_get_ticket_numbers' ) ) {

	/**
	 * Get the lottery ticket numbers.
	 *
	 * @return array.
	 */
	function lty_get_ticket_numbers( $product = false, $user_id = false, $post_status = 'all', $order_id = false ) {

		if ( ! is_object( $product ) ) {
			$product = wc_get_product( $product );
		}

		if ( ! is_object( $product ) ) {
			return array();
		}

		// Post statuses.
		$post_statuses = lty_get_lottery_ticket_statuses();
		if ( 'all' == $post_status ) {
			$post_statuses = array_merge( array( 'lty_ticket_pending' ), $post_statuses );
		}

		global $wpdb;
		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );
		$post_query->leftJoin( $wpdb->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
				->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE )
				->whereIn( '`p`.post_status', $post_statuses )
				->where( '`p`.post_parent', $product->get_lottery_id() )
				->where( '`pm`.meta_key', 'lty_ticket_number' )
				->whereBetween( '`p`.post_date_gmt', $product->get_current_start_date_gmt(), $product->get_lty_end_date_gmt() );

		if ( $user_id && $order_id ) {
			$post_query->leftJoin( $wpdb->prefix . 'postmeta', 'pm1', '`p`.`ID` = `pm1`.`post_id`' )
					->where( '`pm1`.meta_key', 'lty_user_id' )
					->where( '`pm1`.meta_value', $user_id );
			$post_query->leftJoin( $wpdb->prefix . 'postmeta', 'pm2', '`p`.`ID` = `pm2`.`post_id`' )
					->where( '`pm2`.meta_key', 'lty_order_id' )
					->where( '`pm2`.meta_value', $order_id );
		} elseif ( $user_id ) {
			$post_query->leftJoin( $wpdb->prefix . 'postmeta', 'pm1', '`p`.`ID` = `pm1`.`post_id`' )
					->where( '`pm1`.meta_key', 'lty_user_id' )
					->where( '`pm1`.meta_value', $user_id );
		} elseif ( $order_id ) {
			$post_query->leftJoin( $wpdb->prefix . 'postmeta', 'pm1', '`p`.`ID` = `pm1`.`post_id`' )
					->where( '`pm1`.meta_key', 'lty_order_id' )
					->where( '`pm1`.meta_value', $order_id );
		}

		return $post_query->fetchCol( '`pm`.meta_value' );
	}
}

if ( ! function_exists( 'lty_get_unique_lottery_ticket_ids' ) ) {

	/**
	 * Get unique lottery ticket ids by user email ids.
	 *
	 * @since 10.1.0
	 * @param object $product Product object.
	 * @param string $status Ticket status.
	 * @global object $wpdb WordPress database.
	 * @return array
	 */
	function lty_get_unique_lottery_ticket_ids( $product, $status = 'all' ) {
		if ( ! lty_is_lottery_product( $product ) ) {
			return array();
		}

		global $wpdb;
		$status = 'all' === $status ? lty_get_lottery_ticket_statuses() : array( $status );

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );
		$post_query->innerJoin( $wpdb->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
				->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE )
				->whereIn( '`p`.post_status', $status )
				->where( '`p`.post_parent', $product->get_lottery_id() )
				->where( '`pm`.meta_key', 'lty_user_email' )
				->groupBy( '`pm`.meta_value' )
				->whereBetween( '`p`.post_date_gmt', $product->get_current_start_date_gmt(), $product->get_lty_end_date_gmt() );

		$ticket_ids = $post_query->fetchCol( '`p`.ID' );

		return lty_check_is_array( $ticket_ids ) ? $ticket_ids : array();
	}
}

if ( ! function_exists( 'lty_get_my_lotteries' ) ) {

	/**
	 * Get My lotteries.
	 *
	 * @return ID (or) bool
	 */
	function lty_get_my_lotteries( $user_id = false, $status = array( 'lty_ticket_buyer' ) ) {
		global $wpdb;

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );

		$post_query->innerJoin( $wpdb->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
				->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE )
				->whereIn( '`p`.post_status', $status )
				->where( '`pm`.meta_key', 'lty_user_id' )
				->where( '`pm`.meta_value', $user_id )
				->groupBy( '`p`.post_parent' );

		$product_ids = $post_query->fetchCol( '`p`.`post_parent`' );

		if ( ! lty_check_is_array( $product_ids ) ) {
			return array();
		}

		return $product_ids;
	}
}

if ( ! function_exists( 'lty_get_my_winner_lotteries' ) ) {

	/**
	 * Get My lotteries.
	 *
	 * @return ID (or) bool
	 */
	function lty_get_my_winner_lotteries( $user_id = false ) {
		global $wpdb;

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );

		$post_query->innerJoin( $wpdb->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
				->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_WINNER_POSTTYPE )
				->whereIn( '`p`.post_status', array( 'lty_publish' ) )
				->where( '`pm`.meta_key', 'lty_user_id' )
				->where( '`pm`.meta_value', $user_id )
				->groupBy( '`p`.post_parent' );

		$product_ids = $post_query->fetchCol( '`p`.`post_parent`' );

		if ( ! lty_check_is_array( $product_ids ) ) {
			return array();
		}

		return $product_ids;
	}
}

if ( ! function_exists( 'lty_get_my_lost_lottery_ticket_from_product_id' ) ) {

	/**
	 * Get My lotteries.
	 *
	 * @return ID (or) bool
	 */
	function lty_get_my_lost_lottery_ticket_from_product_id( $product_ids, $user_id = false ) {
		global $wpdb;

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );

		$post_query->innerJoin( $wpdb->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
				->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE )
				->whereIn( '`p`.post_status', array( 'lty_ticket_buyer' ) )
				->whereIn( '`p`.post_parent', $product_ids )
				->where( '`pm`.meta_key', 'lty_user_id' )
				->where( '`pm`.meta_value', $user_id );

		$lost_ticket_ids = $post_query->fetchCol( '`p`.`ID`' );

		if ( ! lty_check_is_array( $lost_ticket_ids ) ) {
			return array();
		}

		return $lost_ticket_ids;
	}
}

if ( ! function_exists( 'lty_get_lottery_winners_by_product_id' ) ) {

	function lty_get_lottery_winners_by_product_id( $product_id, $from_date = false, $to_date = false ) {

		if ( empty( $product_id ) ) {
			return;
		}

		return lty_get_lottery_winners( false, $product_id, $from_date, $to_date );
	}
}

if ( ! function_exists( 'lty_get_user_winner_ids_by_product_id' ) ) {

	/**
	 * Get the user winner ids based on product id.
	 *
	 * @return array
	 */
	function lty_get_user_winner_ids_by_product_id( $user_id, $product_id, $from_date = false, $to_date = false ) {

		return lty_get_lottery_winners( $user_id, $product_id, $from_date, $to_date );
	}
}

if ( ! function_exists( 'lty_get_lottery_participant_count' ) ) {

	/**
	 * Get lottery participant count.
	 *
	 * @since 1.0.0
	 * @param int    $product_id Product ID.
	 * @param bool   $count Whether to return participant count or user IDs.
	 * @param string $post_status Post status.
	 * @return int|array
	 */
	function lty_get_lottery_participant_count( $product_id, $count = false, $post_status = false ) {
		if ( empty( $product_id ) ) {
			return;
		}

		$product = wc_get_product( $product_id );
		if ( ! is_object( $product ) ) {
			return;
		}

		$post_statuses = lty_get_lottery_ticket_statuses();
		if ( 'all' === $post_status ) {
			$post_statuses = array_merge( array( 'lty_ticket_pending' ), $post_statuses );
		}

		global $wpdb;

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );
		$post_query->leftJoin( $wpdb->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
				->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE )
				->whereIn( '`p`.post_status', $post_statuses )
				->where( '`p`.post_parent', $product->get_lottery_id() )
				->where( '`pm`.meta_key', 'lty_user_email' )
				->whereBetween( '`p`.post_date_gmt', $product->get_current_start_date_gmt(), $product->get_lty_end_date_gmt() );

		$user_ids = $post_query->fetchCol( 'DISTINCT `pm`.meta_value' );

		return $count ? count( $user_ids ) : $user_ids;
	}
}

if ( ! function_exists( 'lty_get_placed_lottery_product_ticket_ids' ) ) {

	/**
	 * Get the placed lottery product ticket ids.
	 *
	 * @return array
	 * */
	function lty_get_placed_lottery_product_ticket_ids( $product ) {
		if ( ! is_object( $product ) ) {
			$product = wc_get_product( $product );
		}

		if ( ! is_object( $product ) ) {
			return array();
		}

		return lty_get_lottery_tickets( false, $product->get_id(), $product->get_current_start_date_gmt(), $product->get_lty_end_date_gmt(), true );
	}
}

if ( ! function_exists( 'lty_get_purchased_lottery_product_ticket_ids' ) ) {

	/**
	 * Get the purchased lottery product ticket ids.
	 *
	 * @return array
	 * */
	function lty_get_purchased_lottery_product_ticket_ids( $product, $order_by, $order ) {
		if ( ! is_object( $product ) ) {
			$product = wc_get_product( $product );
		}

		if ( ! is_object( $product ) ) {
			return array();
		}

		return lty_get_lottery_tickets( false, $product->get_id(), $product->get_current_start_date_gmt(), $product->get_lty_end_date_gmt(), false, $order_by, $order );
	}
}

if ( ! function_exists( 'lty_get_lottery_looser_ids' ) ) {

	/**
	 * Get the lottery looser ticket ids.
	 *
	 * @return array
	 * */
	function lty_get_lottery_looser_ticket_ids( $product ) {
		if ( ! is_object( $product ) ) {
			$product = wc_get_product( $product );
		}

		if ( ! is_object( $product ) ) {
			return array();
		}

		return lty_get_lottery_looser_tickets( false, $product->get_id(), $product->get_current_start_date_gmt(), $product->get_lty_end_date_gmt(), 'lty_ticket_buyer' );
	}
}


if ( ! function_exists( 'lty_get_lottery_looser_tickets' ) ) {

	/**
	 * Get the lottery looser tickets.
	 *
	 * @return array
	 * */
	function lty_get_lottery_looser_tickets( $user_id = false, $product_id = false, $from_date = false, $end_date = false, $post_status = false ) {

		$args = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE,
			'post_status'    => lty_get_lottery_ticket_statuses(),
			'posts_per_page' => '-1',
			'fields'         => 'ids',
			'orderby'        => 'ID',
		);

		if ( false !== $user_id ) {
			$args['meta_key']   = 'lty_user_id';
			$args['meta_value'] = $user_id;
		}

		if ( false !== $product_id ) {
			$args['post_parent'] = $product_id;
		}

		if ( false !== $from_date || false !== $end_date ) {
			$args['date_query'] = array();

			if ( false !== $from_date ) {
				$args['date_query'][] = array(
					'column' => 'post_date_gmt',
					'after'  => $from_date,
				);
			}

			if ( false !== $end_date ) {
				$args['date_query'][] = array(
					'column' => 'post_date_gmt',
					'before' => $end_date,
				);
			}
		}

		if ( $post_status ) {
			$args['post_status'] = $post_status;
		}

		return get_posts( $args );
	}
}

if ( ! function_exists( 'lty_check_is_ticket_number_exists' ) ) {

	/**
	 * Check lottery ticket was already exist or not
	 *
	 * @return bool
	 * */
	function lty_check_is_ticket_number_exists( $ticket_numbers, $lottery_product_id, $start_date = false, $end_date = false ) {
		global $wpdb;

		if ( empty( $ticket_numbers ) || empty( $lottery_product_id ) ) {
			return array();
		}

		$product = wc_get_product( $lottery_product_id );
		if ( ! lty_is_lottery_product( $product ) ) {
			return array();
		}

		$start_date = ( $start_date ) ? $start_date : $product->get_current_start_date_gmt();
		$end_date   = ( $end_date ) ? $end_date : $product->get_lty_end_date_gmt();

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );
		$post_query->innerJoin( $wpdb->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
				->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE )
				->whereIn( '`p`.post_status', lty_get_ticket_statuses() )
				->where( '`p`.post_parent', $product->get_lottery_id() )
				->where( '`pm`.meta_key', 'lty_ticket_number' )
				->whereIn( '`pm`.meta_value', $ticket_numbers )
				->whereBetween( '`p`.post_date_gmt', $start_date, $end_date );

		$lost_ticket_ids = $post_query->fetchCol( '`p`.`ID`' );
		if ( ! lty_check_is_array( $lost_ticket_ids ) ) {
			return array();
		}

		return $lost_ticket_ids;
	}
}
if ( ! function_exists( 'lty_get_current_winner_user_ids' ) ) {

	/**
	 * Get current winner user ids.
	 *
	 * @return array
	 * */
	function lty_get_current_winner_user_ids( $product_id, $from_date ) {

		$product = wc_get_product( $product_id );
		if ( ! lty_is_lottery_product( $product ) ) {
			return array();
		}

		global $wpdb;
		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );
		$post_query->innerJoin( $wpdb->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
				->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_WINNER_POSTTYPE )
				->whereIn( '`p`.post_status', array( 'lty_publish' ) )
				->where( '`p`.post_parent', $product->get_lottery_id() )
				->whereGt( '`p`.post_date_gmt', $from_date )
				->where( '`pm`.meta_key', 'lty_user_id' );

		return $post_query->fetchCol( 'DISTINCT `pm`.`meta_value`' );
	}
}

if ( ! function_exists( 'lty_get_current_winner_user_emails' ) ) {

	/**
	 * Get current winner user emails.
	 *
	 * @return array
	 * */
	function lty_get_current_winner_user_emails( $product_id, $from_date ) {

		$product = wc_get_product( $product_id );
		if ( ! lty_is_lottery_product( $product ) ) {
			return array();
		}

		global $wpdb;

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );
		$post_query->innerJoin( $wpdb->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
				->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_WINNER_POSTTYPE )
				->whereIn( '`p`.post_status', array( 'lty_publish' ) )
				->where( '`p`.post_parent', $product->get_lottery_id() )
				->whereGt( '`p`.post_date_gmt', $from_date )
				->where( '`pm`.meta_key', 'lty_user_email' );

		return $post_query->fetchCol( 'DISTINCT `pm`.`meta_value`' );
	}
}

if ( ! function_exists( 'lty_get_order_ids_without_tickets' ) ) {

	/**
	 * Get order ids without tickets.
	 *
	 * @return array
	 * */
	function lty_get_order_ids_without_tickets( $product, $order_status = false ) {

		if ( ! lty_is_lottery_product( $product ) ) {
			return array();
		}

		$selected_order_statuses = ! empty( get_option( 'lty_settings_lottery_complete_order_statuses' ) ) ? get_option( 'lty_settings_lottery_complete_order_statuses' ) : array( 'processing', 'completed' );
		$order_statuses          = array();
		foreach ( $selected_order_statuses as $status_key ) {
			$order_statuses[] = 'wc-' . $status_key;
		}

		if ( $order_status ) {
			$order_statuses = array( $order_status );
		}

		global $wpdb;
		$db               = &$wpdb;
		$meta_value_query = '3' != get_option( 'lty_settings_guest_user_participate_type' ) ? 'AND meta.meta_value > 0' : '';
		$order_ids        = $db->get_col(
			$db->prepare(
				"SELECT DISTINCT posts.ID
                                    FROM $db->posts as posts
                                    LEFT JOIN {$db->postmeta} AS meta ON posts.ID = meta.post_id
                                    LEFT JOIN {$db->postmeta} AS meta1 ON posts.ID = meta1.post_id AND meta1.meta_key='lty_ticket_ids_in_order'
                                    LEFT JOIN {$db->prefix}woocommerce_order_items AS order_items ON posts.ID = order_items.order_id
                                    LEFT JOIN {$db->prefix}woocommerce_order_itemmeta AS order_itemmeta ON order_items.order_item_id = order_itemmeta.order_item_id    
                                    WHERE posts.post_type           = 'shop_order'
                                    AND   posts.post_status IN ('" . implode( "','", $order_statuses ) . "')
                                    AND   meta.meta_key             = '_customer_user'    
                                    $meta_value_query
                                    AND   order_itemmeta.meta_key   = '_product_id'
                                    AND   order_itemmeta.meta_value = %d
                                    AND   posts.post_date_gmt BETWEEN %s AND %s
                                    AND   meta1.meta_key is null ORDER BY posts.ID DESC",
				$product->get_id(),
				$product->get_current_start_date_gmt(),
				$product->get_lty_end_date_gmt()
			)
		);

		return $order_ids;
	}
}

if ( ! function_exists( 'lty_get_ticket_id_by_ticket_number' ) ) {

	/**
	 * Get the ticket ID by ticket number.
	 *
	 * @param type $ticket_number
	 */
	function lty_get_ticket_id_by_ticket_number( $ticket_number, $product_id = false ) {
		$args = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE,
			'post_status'    => lty_get_lottery_ticket_statuses(),
			'posts_per_page' => '1',
			'fields'         => 'ids',
			'meta_key'       => 'lty_ticket_number',
			'meta_value'     => $ticket_number,
		);

		if ( $product_id ) {
			$args['post_parent'] = $product_id;
		}

		$ticket_id = get_posts( $args );
		if ( ! lty_check_is_array( $ticket_id ) ) {
			return $ticket_id;
		}

		return reset( $ticket_id );
	}
}

if ( ! function_exists( 'lty_get_lottery_winning_dates' ) ) {

	/**
	 * Get the lottery winning dates.
	 *
	 * @since 8.0.0
	 * @global object $wpdb
	 * @param string $order
	 * @param string $start_date
	 * @return array
	 */
	function lty_get_lottery_winning_dates( $order, $start_date = false ) {
		global $wpdb;

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );
		$post_query->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_WINNER_POSTTYPE )
				->whereIn( '`p`.post_status', array( 'lty_publish' ) )
				->orderBy( 'DATE(`p`.post_date_gmt)', $order )
				->groupBy( 'DATE(`p`.post_date_gmt)' );

		if ( $start_date ) {
			$post_query->whereGte( 'DATE(`p`.post_date_gmt)', $start_date );
		}

		return $post_query->fetchCol( 'DATE(`p`.post_date_gmt) as date' );
	}
}

if ( ! function_exists( 'lty_get_lottery_instant_winning_dates' ) ) {

	/**
	 * Get the lottery instant winning dates.
	 *
	 * @since 9.2.0
	 * @global object $wpdb WordPress database.
	 * @param string $order Whether to order ASC|DESC.
	 * @param string $start_date Start date.
	 * @return array
	 */
	function lty_get_lottery_instant_winning_dates( $order, $start_date = false ) {
		global $wpdb;

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );
		$post_query->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_INSTANT_WINNER_LOG_POSTTYPE )
				->whereIn( '`p`.post_status', array( 'lty_won' ) )
				->orderBy( 'DATE(`p`.post_date_gmt)', $order )
				->groupBy( 'DATE(`p`.post_date_gmt)' );

		if ( $start_date ) {
			$post_query->whereGte( 'DATE(`p`.post_date_gmt)', $start_date );
		}

		return $post_query->fetchCol( 'DATE(`p`.post_date_gmt) as date' );
	}
}

if ( ! function_exists( 'lty_get_instant_winner_rule_statuses' ) ) {

	/**
	 * Get lottery instant winner rule statuses.
	 *
	 * @since 8.0.0
	 * @return array
	 * */
	function lty_get_instant_winner_rule_statuses() {
		/**
		 * This hook is used to alter the instant winner rule statuses.
		 *
		 * @since 8.0.0
		 */
		return apply_filters( 'lty_get_instant_winner_rule_statuses', array( 'publish' ) );
	}
}

if ( ! function_exists( 'lty_get_instant_winner_log_statuses' ) ) {

	/**
	 * Get lottery instant winner log statuses.
	 *
	 * @since 8.0.0
	 * @return array
	 * */
	function lty_get_instant_winner_log_statuses() {
		/**
		 * This hook is used to alter the instant winner log statuses.
		 *
		 * @since 8.0.0
		 */
		return apply_filters( 'lty_get_instant_winner_log_statuses', array( 'lty_available', 'lty_pending', 'lty_won' ) );
	}
}

if ( ! function_exists( 'lty_create_new_instant_winner_rule' ) ) {

	/**
	 * Create New Instant Winner Rule.
	 *
	 * @since 8.0.0
	 * @return integer/string
	 * */
	function lty_create_new_instant_winner_rule( $meta_args, $post_args = array() ) {
		$object = new LTY_Instant_Winner_Rule();
		$id     = $object->create( $meta_args, $post_args );
		return $id;
	}
}

if ( ! function_exists( 'lty_update_instant_winner_rule' ) ) {

	/**
	 * Update Instant Winner rule.
	 *
	 * @since 8.0.0
	 * @return object
	 * */
	function lty_update_instant_winner_rule( $id, $meta_args, $post_args = array() ) {
		$object = new LTY_Instant_Winner_Rule( $id );
		$object->update( $meta_args, $post_args );

		return $object;
	}
}

if ( ! function_exists( 'lty_delete_instant_winner_rule' ) ) {

	/**
	 * Delete Instant Winner rule.
	 *
	 * @since 8.0.0
	 * @return bool
	 * */
	function lty_delete_instant_winner_rule( $id, $force = true ) {
		wp_delete_post( $id, $force );
		return true;
	}
}

if ( ! function_exists( 'lty_get_instant_winner_rule' ) ) {

	/**
	 * Get Instant Winner rule object.
	 *
	 * @since 8.0.0
	 * @return object
	 * */
	function lty_get_instant_winner_rule( $id ) {
		$object = new LTY_Instant_Winner_Rule( $id );
		return $object;
	}
}

if ( ! function_exists( 'lty_get_rule_id_by_ticket_number' ) ) {

	/**
	 * Get rule id by ticket number.
	 *
	 * @since 8.0.0
	 * @return string/bool
	 * */
	function lty_get_rule_id_by_ticket_number( $lottery_id, $ticket_number ) {
		$args     = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_INSTANT_WINNER_RULE_POSTTYPE,
			'post_status'    => lty_get_instant_winner_rule_statuses(),
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'meta_key'       => 'lty_ticket_number',
			'meta_value'     => $ticket_number,
			'post_parent'    => $lottery_id,
		);
		$rule_ids = get_posts( $args );

		return lty_check_is_array( $rule_ids ) ? reset( $rule_ids ) : false;
	}
}

if ( ! function_exists( 'lty_get_instant_winner_rule_ids' ) ) {

	/**
	 * Get instant winner rule ids.
	 *
	 * @since 8.0.0
	 * @return array
	 * */
	function lty_get_instant_winner_rule_ids( $lottery_id, $offset = 0, $limit = -1 ) {
		if ( ! $lottery_id ) {
			return array();
		}

		$args = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_INSTANT_WINNER_RULE_POSTTYPE,
			'post_status'    => lty_get_instant_winner_rule_statuses(),
			'posts_per_page' => $limit,
			'fields'         => 'ids',
			'orderby'        => 'ID',
			'order'          => 'ASC',
			'post_parent'    => $lottery_id,
			'offset'         => $offset,
		);

		return get_posts( $args );
	}
}

if ( ! function_exists( 'lty_get_ticket_id_by_rule_id' ) ) {

	/**
	 * Get ticket id by rule id.
	 *
	 * @since 8.0.0
	 * @return string/bool
	 * */
	function lty_get_ticket_id_by_rule_id( $rule_id ) {
		$args     = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE,
			'post_status'    => array_merge( array( 'lty_ticket_pending' ), (array) lty_get_lottery_ticket_statuses() ),
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'meta_key'       => 'lty_rule_id',
			'meta_value'     => $rule_id,
		);
		$rule_ids = get_posts( $args );
		return lty_check_is_array( $rule_ids ) ? reset( $rule_ids ) : false;
	}
}

if ( ! function_exists( 'lty_create_new_instant_winner_log' ) ) {

	/**
	 * Create new instant winner log.
	 *
	 * @since 8.0.0
	 * @return integer/string
	 * */
	function lty_create_new_instant_winner_log( $meta_args, $post_args = array() ) {
		$object = new LTY_Instant_Winner_Log();
		$id     = $object->create( $meta_args, $post_args );
		return $id;
	}
}

if ( ! function_exists( 'lty_update_instant_winner_log' ) ) {

	/**
	 * Update instant winner log.
	 *
	 * @since 8.0.0
	 * @return object
	 * */
	function lty_update_instant_winner_log( $id, $meta_args, $post_args = array() ) {
		$object = new LTY_Instant_Winner_Log( $id );
		$object->update( $meta_args, $post_args );

		return $object;
	}
}

if ( ! function_exists( 'lty_delete_instant_winner_log' ) ) {

	/**
	 * Delete instant winner log.
	 *
	 * @since 8.0.0
	 * @return bool
	 * */
	function lty_delete_instant_winner_log( $id, $force = true ) {
		wp_delete_post( $id, $force );
		return true;
	}
}

if ( ! function_exists( 'lty_get_instant_winner_log' ) ) {

	/**
	 * Get instant winner log.
	 *
	 * @since 8.0.0
	 * @return array
	 * */
	function lty_get_instant_winner_log( $log_id ) {
		$object = new LTY_Instant_Winner_Log( $log_id );
		return $object;
	}
}

if ( ! function_exists( 'lty_get_instant_winner_log_ids' ) ) {

	/**
	 * Get the instant winner log ids.
	 *
	 * @since 8.0.0
	 * @param int          $lottery_id
	 * @param int          $rule_id
	 * @param int          $relist_count
	 * @param array/string $statuses
	 * @return array
	 */
	function lty_get_instant_winner_log_ids( $lottery_id, $rule_id = false, $relist_count = false, $statuses = 'all' ) {
		if ( ! $lottery_id ) {
			return array();
		}

		if ( 'all' === $statuses ) {
			$statuses = lty_get_instant_winner_log_statuses();
		}

		$args = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_INSTANT_WINNER_LOG_POSTTYPE,
			'post_status'    => $statuses,
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'orderby'        => 'ID',
			'order'          => 'ASC',
			'meta_query'     => array(
				'relation' => 'AND',
				array(
					'key'   => 'lty_lottery_id',
					'value' => $lottery_id,
				),
			),
		);

		if ( false !== $relist_count ) {
			$args['meta_query'][] = array(
				'key'   => 'lty_current_relist_count',
				'value' => $relist_count,
			);
		}

		if ( $rule_id ) {
			$args['post_parent'] = $rule_id;
		}

		$log_ids = get_posts( $args );

		return lty_check_is_array( $log_ids ) ? $log_ids : array();
	}
}

if ( ! function_exists( 'lty_get_instant_winner_log_ids_by_user_id' ) ) {

	/**
	 * Get the instant winner log ids by user ID.
	 *
	 * @since 9.8.0
	 * @param int $user_id User ID.
	 * @return array
	 */
	function lty_get_instant_winner_log_ids_by_user_id( $user_id ) {
		if ( ! $user_id ) {
			return array();
		}

		return get_posts(
			array(
				'post_type'      => LTY_Register_Post_Types::LOTTERY_INSTANT_WINNER_LOG_POSTTYPE,
				'post_status'    => lty_get_instant_winner_log_statuses(),
				'posts_per_page' => -1,
				'fields'         => 'ids',
				'meta_key'       => 'lty_user_id',
				'meta_value'     => $user_id,
			)
		);
	}
}

if ( ! function_exists( 'lty_get_lottery_winner_ids_by_date' ) ) {

	/**
	 * Get the lottery winner IDs by date.
	 *
	 * @since 8.0.0
	 * @global object $wpdb
	 * @param string $date
	 * @return array
	 */
	function lty_get_lottery_winner_ids_by_date( $date ) {
		global $wpdb;

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );
		$post_query->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_WINNER_POSTTYPE )
				->whereIn( '`p`.post_status', array( 'lty_publish' ) )
				->where( 'DATE(`p`.post_date_gmt)', $date );

		return $post_query->fetchCol( '`p`.ID' );
	}
}

if ( ! function_exists( 'lty_get_lottery_instant_winner_log_ids_by_date' ) ) {

	/**
	 * Get the lottery nstant winner log IDs by date.
	 *
	 * @since 8.0.0
	 * @global object $wpdb WordPress database.
	 * @param string $date Date of the lottery instant winner.
	 * @return array
	 */
	function lty_get_lottery_instant_winner_log_ids_by_date( $date ) {
		global $wpdb;

		$post_query = new LTY_Query( $wpdb->prefix . 'posts', 'p' );
		$post_query->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_INSTANT_WINNER_LOG_POSTTYPE )
				->whereIn( '`p`.post_status', array( 'lty_won' ) )
				->where( 'DATE(`p`.post_date_gmt)', $date );

		return $post_query->fetchCol( '`p`.ID' );
	}
}

if ( ! function_exists( 'lty_get_instant_winner_log_id_by_rule_id' ) ) {

	/**
	 * Get instant winner log id by rule id.
	 *
	 * @since 8.0.0
	 * @param string $rule_id
	 * @param string $relist_count
	 * @return string
	 * */
	function lty_get_instant_winner_log_id_by_rule_id( $rule_id, $relist_count ) {
		if ( ! $rule_id ) {
			return false;
		}

		$args = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_INSTANT_WINNER_LOG_POSTTYPE,
			'post_status'    => lty_get_instant_winner_log_statuses(),
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'orderby'        => 'ID',
			'post_parent'    => $rule_id,
			'meta_key'       => 'lty_current_relist_count',
			'meta_value'     => $relist_count,
		);

		$log_ids = get_posts( $args );

		return lty_check_is_array( $log_ids ) ? reset( $log_ids ) : false;
	}
}

if ( ! function_exists( 'lty_get_instant_winner_log_id_by_ticket_id' ) ) {

	/**
	 * Get instant winner log id by ticket id.
	 *
	 * @since 8.0.0
	 * @param string $ticket_id
	 * @return string
	 * */
	function lty_get_instant_winner_log_id_by_ticket_id( $ticket_id ) {
		if ( ! $ticket_id ) {
			return false;
		}

		$args = array(
			'post_type'      => LTY_Register_Post_Types::LOTTERY_INSTANT_WINNER_LOG_POSTTYPE,
			'post_status'    => lty_get_instant_winner_log_statuses(),
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'orderby'        => 'ID',
			'meta_key'       => 'lty_ticket_id',
			'meta_value'     => $ticket_id,
		);

		$log_ids = get_posts( $args );

		return lty_check_is_array( $log_ids ) ? reset( $log_ids ) : false;
	}
}

if ( ! function_exists( 'lty_get_lottery_ids' ) ) {

	/**
	 * Get the Lottery IDs.
	 *
	 * @since 8.5.0
	 * @return array
	 * */
	function lty_get_lottery_ids() {
		static $lottery_ids;

		if ( isset( $lottery_ids ) ) {
			return $lottery_ids;
		}

		$args = array(
			'post_type'      => 'product',
			'post_status'    => array( 'publish', 'private' ),
			'posts_per_page' => '-1',
			'fields'         => 'ids',
			'tax_query'      => array(
				array(
					'taxonomy' => 'product_type',
					'field'    => 'name',
					'terms'    => 'lottery',
				),
			),
		);

		$lottery_ids = get_posts( $args );

		return lty_check_is_array( $lottery_ids ) ? $lottery_ids : array();
	}

}

if ( ! function_exists( 'lty_get_purchased_tickets_ids_on_ticket_logs_tab' ) ) {

	/**
	 * Get the purchased tickets ids on ticket logs tab
	 *
	 * @since 1.0.0
	 * @param object $product Product object.
	 * @param string $search Search term.
	 * @return array
	 */
	function lty_get_purchased_tickets_ids_on_ticket_logs_tab( $product, $search = false ) {
		if ( ! lty_is_lottery_product( $product ) ) {
			return array();
		}

		$is_alphanumeric = false;
		if ( '1' == $product->get_lty_ticket_generation_type() && $product->is_automatic_random_ticket() ) {
			$is_alphanumeric = ! empty( get_option( 'lty_settings_ticket_prefix' ) ) || ! empty( get_option( 'lty_settings_ticket_suffix' ) );
			$is_alphanumeric = ( '2' == get_option( 'lty_settings_generate_ticket_type' ) ) ? true : $is_alphanumeric;
		} else {
			$is_alphanumeric = ! empty( $product->get_lty_ticket_prefix() ) || ! empty( $product->get_lty_ticket_suffix() );
		}

		$key      = $is_alphanumeric ? 'meta_value meta_value_num' : 'meta_value_num';
		$order_by = '1' == get_option( 'lty_settings_single_product_tab_details_order_by', 1 ) ? 'ID' : $key;
		$order    = '1' == get_option( 'lty_settings_single_product_tab_details_order', 1 ) ? 'DESC' : 'ASC';

		$ticket_ids = $product->get_purchased_ticket_ids( $order_by, $order );

		// Search.
		if ( $search ) {
			global $wpdb;
			$searched_ids = array();
			$terms        = explode( ' , ', $search );
			$database     = &$wpdb;

			foreach ( $terms as $term ) {
				$term       = $database->esc_like( $term );
				$post_query = new LTY_Query( $database->prefix . 'posts', 'p' );
				$post_query->select( 'DISTINCT `p`.ID' )
						->leftJoin( $database->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`' )
						->where( '`p`.post_type', LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE )
						->whereIn( '`p`.post_status', lty_get_ticket_statuses() )
						->where( '`p`.post_parent', $product->get_id() )
						->whereIn( '`pm`.meta_key', lty_get_ticket_logs_search_columns() )
						->whereLike( '`pm`.meta_value', lty_get_formatted_ticket_logs_search_term( $term ) );

				$ids = $post_query->fetchCol( 'ID' );

				$searched_ids = array_merge( $searched_ids, $ids );
			}

			$ticket_ids = array_intersect( $ticket_ids, array_filter( array_unique( $searched_ids ) ) );
		}

		return $ticket_ids;
	}
}

if ( ! function_exists( 'lty_get_lottery_ticket_ids_by_order_id' ) ) {

	/**
	 * Get the lottery ticket IDs by order ID.
	 *
	 * @since 9.5.0
	 * @param int $order_id Order ID.
	 * @param int $product_id Product ID.
	 * @return array
	 * */
	function lty_get_lottery_ticket_ids_by_order_id( $order_id, $product_id ) {
		if ( ! $product_id || ! $order_id ) {
			return array();
		}

		return get_posts(
			array(
				'post_type'      => LTY_Register_Post_Types::LOTTERY_TICKET_POSTTYPE,
				'post_status'    => lty_get_lottery_ticket_statuses(),
				'posts_per_page' => '-1',
				'fields'         => 'ids',
				'orderby'        => 'ID',
				'post_parent'    => $product_id,
				'meta_key'       => 'lty_order_id',
				'meta_value'     => $order_id,
			)
		);
	}
}
