<?php
/**
 * Customer - Lottery Multiple Winner
 *
 * @since 1.0.0
 * */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'LTY_Customer_Lottery_Multiple_Winner_Notification' ) ) {

	/**
	 * Class.
	 * */
	class LTY_Customer_Lottery_Multiple_Winner_Notification extends LTY_Notifications {

		/**
		 * Class Constructor.
		 * */
		public function __construct() {
			$this->id          = 'customer_multiple_winner';
			$this->type        = 'customer';
			$this->title       = __( 'Customer - Lottery Multiple Winner', 'lottery-for-woocommerce' );
			$this->description = __( 'Send email to the lottery multiple winners.', 'lottery-for-woocommerce' );

			// Triggers for this email.
			add_action( sanitize_key( $this->plugin_slug . '_lottery_product_after_finished' ), array( $this, 'trigger' ), 10, 1 );
			// Render email shortcode information.
			add_action( 'woocommerce_admin_field_lty_display_email_shortcode_' . sanitize_title( $this->id ), array( $this, 'render_email_shortcode_information' ) );

			parent::__construct();
		}

		/**
		 * Default Subject.
		 * */
		public function get_default_subject() {
			return '{site_name} - Congratulations! You are one of the Winners for a Lottery';
		}

		/**
		 * Default Message.
		 * */
		public function get_default_message() {
			return 'Hi,

			Congratulations! You are one of the winners for the {product_name} Lottery held between {lottery_start_date} - {lottery_end_date}.
			Thanks.';
		}

		/**
		 * Trigger the sending of this email.
		 *
		 * @since 1.0.0
		 * @param int   $product_id Product ID.
		 * @return void
		 */
		public function trigger( $product_id ) {
			$product = wc_get_product( $product_id );
			if ( ! $this->is_valid( $product ) ) {
				return;
			}

			$winner_ids = $product->get_current_winner_ids();
			if ( ! lty_check_is_array( $winner_ids ) ) {
				return;
			}

			foreach ( $winner_ids as $winner_id ) {
				$winner = lty_get_lottery_winner( $winner_id );
				if ( ! $winner->exists() ) {
					continue;
				}

				$this->recipient                            = $winner->get_user_email();
				$this->placeholders['{user_name}']          = $winner->get_user_name();
				$this->placeholders['{first_name}']         = is_object( $winner->get_user() ) ? $winner->get_user()->first_name : '';
				$this->placeholders['{last_name}']          = is_object( $winner->get_user() ) ? $winner->get_user()->last_name : '';
				$this->placeholders['{product_name}']       = $winner->get_product_name( true );
				$this->placeholders['{site_name}']          = $this->get_blogname();
				$this->placeholders['{lottery_start_date}'] = esc_html( LTY_Date_Time::get_wp_format_datetime_from_gmt( $product->get_lty_start_date_gmt() ) );
				$this->placeholders['{lottery_end_date}']   = esc_html( LTY_Date_Time::get_wp_format_datetime_from_gmt( $product->get_lty_end_date_gmt() ) );
				$this->placeholders['{ticket_number}']      = $winner->get_lottery_ticket_number();
				$this->placeholders['{winning_item}']       = lty_get_winner_gift_products_title( array_unique( $winner->get_gift_products() ), $product );

				if ( $this->get_recipient() ) {
					$this->send_email( $this->get_recipient(), $this->get_subject(), $this->get_formatted_message(), $this->get_headers(), $this->get_attachments() );
				}
			}
		}

		/**
		 * Is valid to sent email?.
		 *
		 * @return bool
		 * */
		public function is_valid( $product ) {
			if ( ! $this->is_enabled() ) {
				return false;
			}

			if ( ! is_object( $product ) ) {
				return false;
			}

			if ( 'lty_lottery_finished' !== $product->get_lty_lottery_status() ) {
				return false;
			}

			if ( $product->get_lty_winners_count() == 1 ) {
				return false;
			}

			return true;
		}

		/**
		 * Get the settings array.
		 * */
		public function get_settings_array() {
			$section_fields = array();

			// Customer Lottery Multiple Winner Section Start.
			$section_fields[] = array(
				'type'  => 'title',
				'title' => __( 'Customer - Lottery Multiple Winner', 'lottery-for-woocommerce' ),
				'id'    => 'lty_lottery_multiple_winner_email_options',
			);
			$section_fields[] = array(
				'title'   => __( 'Enable', 'lottery-for-woocommerce' ),
				'type'    => 'checkbox',
				'default' => 'no',
				'id'      => $this->get_option_key( 'enabled' ),
			);
			$section_fields[] = array(
				'title'   => __( 'Subject', 'lottery-for-woocommerce' ),
				'type'    => 'text',
				'default' => $this->get_default_subject(),
				'id'      => $this->get_option_key( 'subject' ),
			);
			$section_fields[] = array(
				'title'     => __( 'Message', 'lottery-for-woocommerce' ),
				'type'      => 'lty_custom_fields',
				'lty_field' => 'wpeditor',
				'default'   => $this->get_default_message(),
				'id'        => $this->get_option_key( 'message' ),
			);
			$section_fields[] = array(
				'type' => 'lty_display_email_shortcode_' . $this->id,
			);
			$section_fields[] = array(
				'type' => 'sectionend',
				'id'   => 'lty_lottery_multiple_winner_email_options',
			);
			// Customer Lottery Multiple Winner Email Section End.

			return $section_fields;
		}

		/**
		 * Render email shortcode information.
		 * */
		public function render_email_shortcode_information() {
			$shortcodes_info = array(
				'{user_name}'          => array(
					'description' => __( 'Displays the lottery user name', 'lottery-for-woocommerce' ),
				),
				'{product_name}'       => array(
					'description' => __( 'Displays the lottery product name', 'lottery-for-woocommerce' ),
				),
				'{first_name}'         => array(
					'description' => __( 'Displays the lottery first name', 'lottery-for-woocommerce' ),
				),
				'{last_name}'          => array(
					'description' => __( 'Displays the lottery last name', 'lottery-for-woocommerce' ),
				),
				'{lottery_start_date}' => array(
					'description' => __( 'Displays the lottery start date', 'lottery-for-woocommerce' ),
				),
				'{lottery_end_date}'   => array(
					'description' => __( 'Displays the lottery end date', 'lottery-for-woocommerce' ),
				),
				'{Ticket_number}'      => array(
					'description' => __( 'Displays the lottery ticket number', 'lottery-for-woocommerce' ),
				),
				'{winning_item}'       => array(
					'description' => __( 'Displays the lottery winning prize details', 'lottery-for-woocommerce' ),
				),
				'{site_name}'          => array(
					'description' => __( 'Displays the site name', 'lottery-for-woocommerce' ),
				),
				'{logo}'               => array(
					'description' => __( 'Displays the logo', 'lottery-for-woocommerce' ),
				),
			);

			include_once LTY_ABSPATH . 'inc/admin/menu/views/html-email-shortcodes-info.php';
		}
	}

}
